#!/usr/bin/perl
#
# perl script to return values from a bind query log to mrtg
#
# author: Alexander Schreiber <als@thangorodrim.de>
#
# version: $Id: bind_queries.pl,v 1.2 2003/04/20 18:52:57 als Exp $
#
# usage: bind_queries.pl query_log state_file
# with:    query_log  -> bind querylog file, usually /var/log/bin/queries
#          state_file -> file to keep state between checks, default is
#                        /var/tmp/bind_queries.state
#
# example entry for mrtg.cfg:
#
# BodyTag[frodo-bindqueries]: <body  bgcolor="#FDF6E6">
# Target[frodo-bindqueries]: `/opt/mrtgsensors/bind_queries.pl /var/log/bind/queries /var/tmp/bind_queries.state`
# Title[frodo-bindqueries]:  frodo's bind queries
# MaxBytes[frodo-bindqueries]: 1000
# PageTop[frodo-bindqueries]: <H1> frodo's bind queries </H1>
# LegendI[frodo-bindqueries]:  A records
# LegendO[frodo-bindqueries]:  PTR records
# Legend1[frodo-bindqueries]:  A records
# Legend2[frodo-bindqueries]:  PTR records
# YLegend[frodo-bindqueries]: queries
# ShortLegend[frodo-bindqueries]: queries
# Options[frodo-bindqueries]: gauge, growright, transparent,integer,nopercent
#

use strict;
use warnings;
use diagnostics;

sub print_help {
# print usage help
# parameters: none
# returns: nothing

    print "bind_queries.pl return stats about queries processed by bind\n";
    print "usage: bind_queries.pl query_log state_file\n";
    print "       with query_log  -> bind querylog file, usually\n";
    print "                          /var/log/bin/queries\n";
    print "            state_file -> file to keep state between checks,\n";
    print "                          default is /var/tmp/bind_queries.state\n";
}

sub check_args {
# checks commandline arguments
# parameters: none
# returns: 0 if everything is ok, 1 if not

    my $count;

    $count = scalar(@ARGV);
    if ( $count != 2 ) {
        print "error: wrong number of arguments!\n";
        return 1;
    } else {
        return 0;
    }

}

sub read_query_log {
# reads a bind query log and returns the number of queries since last read
# parameters: $query_log -> path to bind query log file
#             $state_log -> path to state log file
# returns: ($in_a, $in_ptr) with $in_a   -> number of A record queries,
#                                $in_ptr -> number of PTR record queries

    my $query_log = shift;
    my $state_log = shift;


    my $line;
    my $last_line;
    my $saved_last;

    my ($in_a, $in_ptr) = (0, 0);


    open(QUERY, "<$query_log") or die "cannot open $query_log";

# handle non-existing state file
    unless ( -f $state_log ) {
        $last_line = '#############################################';
    } else {
        open(STATE, "<$state_log") or die "cannot open $state_log";
        $last_line = <STATE>;
        close(STATE);
        chomp($last_line);
    }

# simple approach: We read the file from beginning to end, counting all A
# and PTR queries. If we hit a line matching $last_line, the counters are
# reset. If we hit it, we read a file that we have read before and from the
# reset on our counters will be correct, if we don't hit it, then the file
# has been rotated since last read and our counters are as correct as possible
# in this situation.

    while ( $line = <QUERY> ) {
        chomp($line);
        if ( $line =~ / IN PTR$/ ) { # PTR record
            $in_ptr++;
        }
        if ( $line =~ / IN A$/ ) { # A record
            $in_a++;
        }
        if ( $line eq $last_line ) {
            ($in_a, $in_ptr) = (0, 0);
        }
        $saved_last = $line;
    }

    close(QUERY);
# save state
    open(STATE, ">$state_log") or die "cannot open $state_log for writing";
    print STATE "$saved_last\n";
    close(STATE);

    return ($in_a, $in_ptr);
}

################### main ################

my ($in_a, $in_ptr);

unless ( &check_args() ) {
    ($in_a, $in_ptr) = &read_query_log($ARGV[0], $ARGV[1]);
    print "$in_a\n$in_ptr\n0\n";
} else {
    &print_help();
    exit(1);
}
